package furny.ga;

import org.apache.commons.lang3.ObjectUtils;

import com.jme3.math.FastMath;

/**
 * The room vector is a tuple of x- and y-coordinates and rotation steps. It is
 * part of the furniture genotype.
 * 
 * @since 11.08.2012
 * @author Stephan Dreyer
 */
public final class RoomVector {
  private static final float ROTATION_DEGREES = 90f;
  private static final int MAX_ROTATION = 4;

  private final int xGene, yGene, rotGene;

  /**
   * Creates a new room vector and sets all fields.
   * 
   * @param xGene
   *          The gene for x (in centimeters).
   * @param yGene
   *          The gene for y (in centimeters).
   * @param rotationSteps
   *          The number of the 90 rotations.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public RoomVector(final int xGene, final int yGene, final int rotationSteps) {
    this.xGene = xGene;
    this.yGene = yGene;
    this.rotGene = rotationSteps % MAX_ROTATION;
  }

  /**
   * Get the gene for x.
   * 
   * @return X gene (centimenters).
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public int getXGene() {
    return xGene;
  }

  /**
   * Get the gene for y.
   * 
   * @return Y gene (centimenters).
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public int getYGene() {
    return yGene;
  }

  /**
   * Get the real x coordinate.
   * 
   * @return X coordinate (meters).
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public float getX() {
    return xGene / 100f;
  }

  /**
   * Get the real y coordinate.
   * 
   * @return Y coordinate (meters).
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public float getY() {
    return yGene / 100f;
  }

  /**
   * Getter for the rotation.
   * 
   * @return The rotation in degrees.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public float getRotation() {
    return FastMath.DEG_TO_RAD * ROTATION_DEGREES * rotGene;
  }

  /**
   * Get the gene for the rotation.
   * 
   * @return Rotation gene (90 rotations).
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public int getRotationSteps() {
    return rotGene;
  }

  /**
   * Creates a new room vector that is translated by x and y.
   * 
   * @param x
   *          Translation x (centimeters).
   * @param y
   *          Translation y (centimeters).
   * @return The new room vector.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public RoomVector getTranslatedInstance(final int x, final int y) {
    return new RoomVector(this.xGene + x, this.yGene + y, rotGene);
  }

  /**
   * Creates a new room vector that is rotated by the given number of 90 steps.
   * 
   * @param rotationSteps
   *          Steps to rotate.
   * @return Rotated room vector.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public RoomVector getRotatedInstance(final int rotationSteps) {
    return new RoomVector(this.xGene, this.yGene, (rotGene + rotationSteps)
        % MAX_ROTATION);
  }

  @Override
  public boolean equals(final Object obj) {
    return obj != null && obj.getClass() == getClass()
        && ((RoomVector) obj).xGene == xGene
        && ((RoomVector) obj).yGene == yGene
        && ((RoomVector) obj).rotGene == rotGene;
  }

  @Override
  public int hashCode() {
    return ObjectUtils.hashCodeMulti(xGene, yGene, rotGene);
  }

  @Override
  protected RoomVector clone() {
    return new RoomVector(xGene, yGene, rotGene);
  }

  /**
   * Calculates the euklidean distance between this an the other room vector.
   * 
   * @param vec
   *          The other room vector.
   * @return distance in centimenters.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public float distance(final RoomVector vec) {
    return FastMath.sqrt(FastMath.pow(getX() - vec.getX(), 2f)
        + FastMath.pow(getY() - vec.getY(), 2f));
  }

  /**
   * Calculates the degrees difference between the rotations.
   * 
   * @param vec
   *          The other room vector.
   * @return The degrees between -180 and +180.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public float diffAngleDeg(final RoomVector vec) {
    float diffangle = (FastMath.RAD_TO_DEG * getRotation() - FastMath.RAD_TO_DEG
        * vec.getRotation()) + 180f;
    diffangle = diffangle / 360f;
    diffangle = ((diffangle - FastMath.floor(diffangle)) * 360f) - 180f;
    return diffangle;
  }

  /**
   * Calculates the rotation steps difference between the rotations.
   * 
   * @param vec
   *          The other room vector.
   * @return The difference between -2 and 2.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public int diffRotation(final RoomVector vec) {
    float diffRot = (rotGene - vec.rotGene + 2f) / 4f;
    diffRot = ((diffRot - FastMath.floor(diffRot)) * 4f) - 2f;
    return Math.round(diffRot);
  }

  @Override
  public String toString() {
    return getClass().getSimpleName() + "(" + xGene + "," + yGene + ","
        + getRotation() + ")";
  }
}
